/*
 * Constraint.java
 *
 * Created on 11. Dezember 2006, 23:53
 *
 * Written by:
 * Frank Bicking: 504628
 * Martin Schrder: 502301
 *
 * Group: Bort
 */

package constraintprop;

import java.lang.Math;

/**
 * A class that serves as the main storage for the constrained Intervals.
 * Constraint propagation itself is only implemented as a back-end here.
 *
 * @author Martin Schrder, 502301
 * @author Frank Bicking, 504628
 */
public class Constraint {
    /** At first, we define the landmark constants. */
    public static final int GoalBlue = 0;
    public static final int GoalYellow = 1;
    public static final int FlagNW = 2;
    public static final int FlagNE = 3;
    public static final int FlagSW = 4;
    public static final int FlagSE = 5;
    public static final int Middle = 6;
    /* END STATIC DEF */
    
    private Interval xPos;
    private Interval yPos;
    private Interval angle;
    
    private int cameraHorizResolution;
    private int cameraVertResolution;
    /**
     * All angles need to be in radians --> -Pi < angle > Pi
     */
    private double cameraHorizAngle;
    private double cameraVertAngle;
    
    /** Creates a new instance of Constraint.
     * Initializes the Intervals with sane values and sets up other trivia,
     * like the camera properties.
     * It also introduces the first two constraints on the XY Values:
     * CX = (X : 0 < X < 6000)
     * CY = (Y : 0 < Y < 4000)
     */
    public Constraint(double cameraHorizAngle, double cameraVertAngle,
                      int cameraHorizResolution, int cameraVertResolution) {
        //Creating the intervals
        xPos = new Interval();
        yPos = new Interval();
        angle = new Interval();
        //Setting the parameters of the camera and converting degrees to radians
        this.cameraHorizResolution = cameraHorizResolution;
        this.cameraVertResolution = cameraVertResolution;
        this.cameraHorizAngle = degToRad(cameraHorizAngle);
        this.cameraVertAngle = degToRad(cameraVertAngle);
        //Adding the constraint by limiting the XY Intervals
        xPos.addInterval(0, 6000);
        yPos.addInterval(0, 4000);
        //An angle should be constrained so as to be between 0 and 360
        angle.addInterval(0.0, degToRad(360.0));
    }
    
    private double degToRad(double angle) {
        return angle * (java.lang.Math.PI * 2 / 360);
    }
    
    /**
     * Calculates the distance to a given object in cm.
     * It needs the identifier of the landmark and apparent size of the
     * landmark in the frame.
     */
    private double distanceTo(int landmark, int size) {
        /* dist = (r) / tan(angle)
         * r = Absolute, real size of the landmark in cm divided by 2
         * angle = (size * cameraHorizAngle) / (2 * cameraHorizResolution)
         */
        double angle = (size * cameraHorizAngle) / (2 * cameraHorizResolution);
        switch (landmark) {
            case GoalBlue   : return 400 / java.lang.Math.tan(angle);
            case GoalYellow : return 5 / java.lang.Math.tan(angle);
            case FlagNW     : return 5 / java.lang.Math.tan(angle);
            case FlagNE     : return 5 / java.lang.Math.tan(angle);
            case FlagSW     : return 5 / java.lang.Math.tan(angle);
            case FlagSE     : return 5 / java.lang.Math.tan(angle);
            case Middle     : return 180 / java.lang.Math.tan(angle);
        }
        //If we receive insane values
        return -1.0;
    }
    
    /**
     * Computes the global rotational angle from the bearing of a given landmark
     * and a given position of the robot. The general formula is:
     * angle = a_offset +- (pos1 / pos2) - bearing
     *
     * The bearing is simply assumed to be already given in radians.
     * A description of the formulas can be found in the solution to exercise A
     * in the appended PDF document.
     */
    private double computeAngle(int landmark, double bearing, double posX, double posY) {
        if (landmark == GoalBlue) {
            if (posY <= 2000) {
                return degToRad(270) - java.lang.Math.atan(posX / (2000 - posY)) - bearing;
            } else {
                return degToRad(90) + java.lang.Math.atan(posX / (2000 + posY)) - bearing;
            }
        } else if (landmark == GoalYellow) {
            if (posY <= 2000) {
                return degToRad(270) + java.lang.Math.atan((6000 - posX) / (2000 - posY)) - bearing;
            } else {
                return degToRad(90) - java.lang.Math.atan((6000 - posX) / (2000 + posY)) - bearing;
            }
        } else if (landmark == FlagNW) {
            if (posX <= 1650) {
                return java.lang.Math.atan(posY / (1650 - posX)) - bearing;
            } else {
                return degToRad(180) - java.lang.Math.atan(posY / (posX - 1650)) - bearing;
            }
        } else if (landmark == FlagNE) {
            if (posX <= 4350) {
                return java.lang.Math.atan(posY / (4350 - posX)) - bearing;
            } else {
                return degToRad(180) - java.lang.Math.atan(posY / (posX - 4350)) - bearing;
            }
        } else if (landmark == FlagSW) {
            if (posX <= 1650) {
                return degToRad(360) - java.lang.Math.atan((4000 - posY) / (1650 - posX)) - bearing;
            } else {
                return degToRad(180) + java.lang.Math.atan((4000 - posY) / (posX - 1650)) - bearing;
            }
        } else if (landmark == FlagSE) {
            if (posX <= 1650) {
                return degToRad(360) - java.lang.Math.atan((4000 - posY) / (4350 - posX)) - bearing;
            } else {
                return degToRad(180) + java.lang.Math.atan((4000 - posY) / (posX - 4350)) - bearing;
            }
        } else return -1.0;
    }
    
    /**
     * Constrains the local intervals by examining the data of a given 
     * landmark with a given bearing-angle, apparent size and error in percent
     * The constraints for the XY-Position are:
     * 
     * C_{XY1} = (X,Y : d(GoalBlue) = \sqrt{X^2 + (2000+Y)^2})
     * C_{XY2} = (X,Y : d(GoalYellow) = \sqrt{(6000+X)^2 + (2000+Y)^2}
     * C_{XY3} = (X,Y : d(FlagNW) = \sqrt{(1650+X)^2 + Y^2}
     * C_{XY4} = (X,Y : d(FlagNE) = \sqrt{(4350+X)^2 + Y^2}
     * C_{XY5} = (X,Y : d(FlagSW) = \sqrt{(1650+X)^2 + (4000+Y)^2}
     * C_{XY6} = (X,Y : d(FlagSE) = \sqrt{(4350+X)^2 + (4000+Y)^2}
     * C_{XY7} = (X,Y : d(Middle) = \sqrt{(3000+X)^2 + (1800+Y)^2}
     *
     * In a perfect world, the constraint would yield us a circle
     * around the landmarks as a constraint for X and Y, but since the
     * Interval class can only take linear values and because we could
     * find a suitable X to every possible Y and vice versa,
     * we get a rectangular interval, which naturally includes positions we
     * couldn't really be at. That explains why the above mentioned constraints
     * only roughly resemble the actual equations used.
     *
     * In retrospect, using three bearing angles to get the current position
     * would've been ohh so much better. But well, hindsight is 20/20.
     */
    public int constrain(int landmark, double bearing, int size, double error) {
        double[] tmp_angle = new double[4];
        double lowAngle = Double.POSITIVE_INFINITY;
        double highAngle = Double.NEGATIVE_INFINITY;
        /* At first, we calculate the distance to the landmark in question and
         * convert it from cm to mm and afterwards apply the error of 10%.
         * Overestimating the distance is always a good idea in that case.
         */
        double dist = distanceTo(landmark, size) * 10 * (1 + (error / 100));
        if (dist < 0) return -1;
        
        //Why no switch-case? Because if-else looks nicer! :)
        if (landmark == GoalBlue) {
            /* As mentioned, in a perfect world, the constraint would yield us
             * a circle around the landmark as a constraint for X and Y, but
             * due to aforementioned reasons, we have to do with rectangular
             * intervals yielded by the following formulas:
             * X = [OffsetX - d(landmark); OffsetX - d(landmark))]
             * Y = [OffsetX - d(landmark); d(landmark) + OffsetY]
             */
            xPos.addInterval(-dist, dist);
            yPos.addInterval(2000 - dist, 2000 + dist);
        } else if (landmark == GoalYellow) {
            xPos.addInterval(6000 - dist, 6000 + dist);
            yPos.addInterval(2000 - dist, 2000 + dist);
        } else if (landmark == FlagNW) {
            xPos.addInterval(1650 - dist, 1650 + dist);
            yPos.addInterval(- dist, dist);
        } else if (landmark == FlagNE) {
            xPos.addInterval(4350 - dist, 4350 + dist);
            yPos.addInterval(- dist, dist);
        } else if (landmark == FlagSW) {
            xPos.addInterval(1650 - dist, 1650 + dist);
            yPos.addInterval(4000 - dist, 4000 + dist);
        } else if (landmark == FlagSE) {
            xPos.addInterval(4350 - dist, 4350 + dist);
            yPos.addInterval(4000 - dist, 4000 + dist);
        } else if (landmark == Middle) {
            xPos.addInterval(3000 - dist, 3000 + dist);
            yPos.addInterval(1800 - dist, 1800 + dist);
        } else return -1;
        /* After we have constrained the XY Interval, we compute the possible
         * angles by computing angles for all four corners of the XY-Interval
         * rectangle. Doing this ensure that we get the broadest interval of
         * angles possible and furthermore allows us to not apply another
         * error-percentage, because we've already applied it once to the
         * position by blurrying the distance.
         */
        tmp_angle[0] = computeAngle(landmark, degToRad(bearing), xPos.getCutLeft(), yPos.getCutLeft());
        tmp_angle[1] = computeAngle(landmark, degToRad(bearing), xPos.getCutLeft(), yPos.getCutRight());
        tmp_angle[2] = computeAngle(landmark, degToRad(bearing), xPos.getCutRight(), yPos.getCutLeft());
        tmp_angle[3] = computeAngle(landmark, degToRad(bearing), xPos.getCutRight(), yPos.getCutRight());
        //And now we find the min and max angle
        for (int i = 0; i < 4; i++) {
            if (tmp_angle[i] < lowAngle) lowAngle = tmp_angle[i];
            if (tmp_angle[i] > highAngle) highAngle = tmp_angle[i];
        }
        //Now, we set an appropriate angle-interval
        angle.addInterval(lowAngle, highAngle);
        //If we reach this here, we have successfully computed the intervals
        return 0;
    }
    
    /**
     * Returns the most middle of the probable X-position interval.
     */
    public double getXPos() {
        return (xPos.getCutLeft() + xPos.getCutRight()) / 2;
    }
    
    /**
     *  Returns the most middle of the probable Y-position interval.
     */
    public double getYPos() {
        return (xPos.getCutLeft() + xPos.getCutRight()) / 2;
    }
    
    /**
     *  Returns the most middle of the probable angle interval.
     */
    public double getAngle() {
        return (angle.getCutLeft() + angle.getCutRight()) / 2;
    }
    
    /**
     * Returns a stringified representation of all intervals by calling the
     * toString() functions if the Intervals
     */
    public String toString() {
        return ("X Interval = " + xPos.toStringCut() +
                "\nY Interval = " + yPos.toStringCut() + 
                "\nAngle Interval = " + angle.toStringCut());
    }
    
}
